/*
________________________________________________________________________________

   03.make-denominator.sas 
   OPA Measure Calculations
   Step 3: Define denominator for measures.

   Summary  : Defines denominator for measure based on population. Creates
   exclusion variables and postpartum population flags, if applicable.

   Authors  : P. Hastings / phil -at- farharbor -dot- com
            : F. Dong / fei -at- farharbor -dot- com 
			: E. Puga / ella -at- farharbor -dot- com   
            : Based on original code by B. Frederiksen -at- DHHS/OPA
   Version  : v3.01
   Date     : 2020-03-20   
   Revisions: None. 
   History  : 2016-12-19(1.00), 2017-02-22(1.01), 2017-05-17(1.02), 
              2018-04-20(1.03), 2018-08-20(1.03a), 2019-02-26 (2.01)
________________________________________________________________________________

*/

*=============================================================================;
*  DEFINE DENOMINATOR FOR MEASURE, CREATE EXCLUSION VARIABLE
*=============================================================================;

*=============================================================================;
*  All women denominator created first
*=============================================================================;

*...Sort ContraceptiveClaims dataset by Patient ID and date of service to prepare a one record per person dataset;
proc sort data = ContraceptiveClaims;
   by PatientID_OPA DOSNew;
   run;

data ContraceptiveClaimsOne; 
   set ContraceptiveClaims; 
   by PatientID_OPA DOSNew;
   if first.PatientID_OPA ;
   run;

*...Report the total number of women of reproductive age and year of original data;
proc freq data = ContraceptiveClaimsOne ORDER=FORMATTED;
   table year /list missing;
   table AgeGroup*year /list missing;
   title2  ;
   title3 "Original Data: Number of Patients by Year and AgeGroup";
   title4 "Year &year. Selected for Analysis";
   title5;
   run;
   title;

*...Restrict ContraceptiveClaims and ContraceptiveClaimsOne datasets to one year;
data ContraceptiveClaims;
   set ContraceptiveClaims (where=(year=&year.));
   run;
data ContraceptiveClaimsOne;
   set ContraceptiveClaimsOne (where=(year=&year.));
   run;


*...This macro creates one record per person datasets with indicators for these conditions: infecund, pregnant, pregnantfinal, non-live-birth, live birth, end of pregnancy;
%macro exclude(condition,firstlast,conditionsuffix); 
   *...Macro parameters: 1) condition and 2) which sorted claim to select (first or last by date of service), 3)suffix for a condition to distinguish sort types...[PH];
   options mprint=1;
   *...Create a dataset with only patients with condition;
   data %cmpres(&condition.&conditionsuffix.);
      set ContraceptiveClaims;
      where &condition.=1;
      run;

   *...Sort claims dataset by Patient ID and date of service to prepare to create a one record per person dataset with women who have condition;
   proc sort data = %cmpres(&condition.&conditionsuffix.); 
      by PatientID_OPA DOSNew;
      run;

   *...Create a one record per person dataset with a condition indicator;
   data %cmpres(&condition.&conditionsuffix.One) (keep = PatientID_OPA %cmpres(&condition.&conditionsuffix.Indi) %cmpres(&condition.&conditionsuffix.Month) %cmpres(&condition.&conditionsuffix.SD));
      set %cmpres(&condition.&conditionsuffix.);
      format %cmpres(&condition.&conditionsuffix.SD) yymmddd10.;
      by PatientID_OPA DOSNew;
      if &firstlast..PatientID_OPA;
      %cmpres(&condition.&conditionsuffix.Indi) = 1;
      %cmpres(&condition.&conditionsuffix.Month) = month;
      %cmpres(&condition.&conditionsuffix.SD) = DOSNew;
      run;

   *...Sort condition indicator dataset to prepare to merge;
   proc sort data = %cmpres(&condition.&conditionsuffix.One); 
      by PatientID_OPA;
      run;
%mend;

%exclude(Infecund,first,); 
%exclude(Pregnant,first,); *...Use first date of any pregnancy claim to flag preg...[PH];
%exclude(NonLiveBirth,last,);
%exclude(LiveBirth,last,);
%exclude(PregEnd,last,);
%exclude(Pregnant,last,Final); *...Experimental: This gets the final claim showing any related pregnancy codes...[PH];

*...Create dataset with pregnancy and end of pregnancy indicators;
data PregnancyIndicator (keep = PatientID_OPA InfecundIndi InfecundMonth InfecundSD 
        PregnantIndi PregnantMonth PregnantSD PregnantFinalIndi PregnantFinalMonth PregnantFinalSD 
        NonLiveBirthIndi NonLiveBirthMonth NonLiveBirthSD 
        LiveBirthSD LiveBirthIndi LiveBirthMonth PregEndIndi PregEndMonth PregEndSD); 
   merge ContraceptiveClaimsOne (in = a) InfecundOne PregnantOne PregnantFinalOne NonLiveBirthOne LiveBirthOne PregEndOne;
   by PatientID_OPA;
   if a = 1;
   run;

*...Sort pregnancy indicator dataset to prepare to merge with multiple records per person full dataset;
proc sort data = PregnancyIndicator; 
   by PatientID_OPA;
   run;
proc sort data = ContraceptiveClaims ; 
   by PatientID_OPA;
   run;

*...Merge pregnancy indicator dataset and multiple records per person full dataset to give women pregnancy indicators;
*...Create exclude variable (exclude = 1 for infecund, exclude = 2 for still pregnant, exclude = 3 for pregnancies ending in last 3 months of the year);
data ContraceptivesPreg;
   merge ContraceptiveClaims PregnancyIndicator; 
   by PatientID_OPA;
   if InfecundIndi = 1 then Exclude_all = 1;
   else if PregnantIndi = 1 and PregEndIndi ^=1 then Exclude_all = 2;
   else if LiveBirthIndi = 1 and LiveBirthMonth > 9 then Exclude_all = 3;
   else Exclude_all=0; 
   run;
*...Delete intermediate datasets;
proc datasets lib = work nolist;
   delete InfecundOne PregnantOne NonLiveBirthOne LiveBirthOne PregEndOne Infecund Pregnant 
          NonLiveBirth LiveBirth PregEnd PregnantFinal PregnantFinalOne pregnancyindicator ContraceptiveClaims;
   quit;

*...Ensure only women ages 15 to 45 years are included in the dataset;
proc sort data = ContraceptivesPreg out = ContraceptivesAge(where=(15 <= AGE_CALC < 45)); 
   by PatientID_OPA DOSNew;
   run;

*...This dataset defines the denominator and will be merged on to the numerator dataset at the end;
data DenominatorDataOne; 
   set ContraceptivesAge;
   by PatientID_OPA DOSNew;
   if first.PatientID_OPA;
   attrib exclude_all label = 'Exclusions: All Women (0=include | 1=infecund | 2=still pregnant | 3=preg ended Oct-Dec)';
   run;

%*...Rename dataset to be used in numerator calculations;
data NumeratorData; 
   set ContraceptivesAge;
   run;

%*...Sort the multiple records per person dataset to prepare for a one record per person dataset;
proc sort data = NumeratorData; 
   by PatientID_OPA DOSNew;
   run;

%*...Create a one record per person dataset to determine how many women were excluded;
data NumeratorDataOne; 
   set NumeratorData;
   by PatientID_OPA DOSNew;
   if last.PatientID_OPA;
   run;


*=============================================================================;
*  END of all women section 
*=============================================================================;



*=============================================================================;
*  POSTPARTUM SECTION
*  Postpartum denominator(s) created next 
*=============================================================================;

%macro postpartum;
%*...If user specifies "yes" for postpartum measure calculations, create exclusion variable for postpartum calculations and flag claims within "n" days postpartum;
%if &postpartum_st. = y %then 
   %do;

   %*...Create a postpartum only dataset with just codes to identify a live birth or delivery from Table PCU-A;
   data LiveBirth_pp; 
      set ContraceptivesPreg;
      where LiveBirth = 1;
      run;

   %*...Sort LiveBirth dataset by Patient ID and DOSNew variable to prepare to create a dataset with women who had a live birth in the measurement year;
   proc sort data = LiveBirth_pp; 
      by PatientID_OPA DOSNew;
      run;

   %*...Create a one record per person dataset with first live births of the measurement year;
   %*...Create a variable for the date of the first live birth of the measurement year, only keep Patient ID and firstlivebirthdate;
   data LiveBirthFirst (keep = PatientID_OPA FirstLiveBirthDate); 
      set LiveBirth_pp; 
      format FirstLiveBirthDate yymmddd10.;
      by PatientID_OPA DOSNew;
      if first.PatientID_OPA;
      FirstLiveBirthDate = DOSNew;
      run;
   %*...Data for identifying date of last live birth in year...[PH];
   data LiveBirthLast (keep = PatientID_OPA LastLiveBirthDate); 
      set LiveBirth_pp; 
      format LastLiveBirthDate yymmddd10.;
      by PatientID_OPA DOSNew;
      if last.PatientID_OPA;
      LastLiveBirthDate = DOSNew;
      run;
   %*...Sort one record per person livebirthfirst dataset by Patient ID to prepare to merge with multiple record per person livebirth dataset;
   proc sort data = LiveBirthFirst; 
      by PatientID_OPA;
      run;
   proc sort data = LiveBirthLast; 
      by PatientID_OPA;
      run;
   data livebirthdates;
      merge LiveBirthFirst LiveBirthLast;
      by PatientID_OPA;
      if LastLiveBirthDate - FirstLiveBirthDate >=180 then numlivebirths=2;
      else numlivebirths=1;
      run;
   %*...Sort multiple record per person livebirth dataset by Patient ID to prepare to merge with livebirthfirst dataset; 
   proc sort data = LiveBirth_pp;
      by PatientID_OPA ;
      run;
   %*...Sort the ContraceptivePreg dataset to prepare to merge with the LiveBirthMergeCount dataset;
   proc sort data = ContraceptivesPreg; 
      by PatientID_OPA DOSnew;
      run;
   %*...Create a new variable called DeliveryDiff to get number of days between date of service of each livebirth claims and firstlivebirthdate;
   %*...Create an indicator called DeliveryCount = 1 that will be applied to each record until a second live birth is identified in subsequent steps;
   %*...Create a new dataset called ContraceptiveClaimsNewID by merging the ContraceptivePreg dataset and the LiveBirthMergeCount dataset;
   %*...If women have a NewID because they had a live birth in the measurement year, the NewID replaces their old Patient ID; 
   %*...which is the same Patient ID with a _1 or _2 at the end.;  
   %*...If they do not have a NewID because they did not have a live birth in the measurement year, the Patient ID remains the same as before;

   data ContraceptiveClaimsNewID; 
      *...Simplify using just one dataset...[PH];
      *...Drop previous pregnancy indicators and recalculate by birth event later...[PH];
      merge livebirthdates (in=liveb keep=PatientID_OPA FirstLiveBirthDate LastLiveBirthDate numlivebirths)  
            ContraceptivesPreg (drop=PregnantIndi PregnantMonth PregnantSD NonLiveBirthIndi NonLiveBirthMonth NonLiveBirthSD 
                                     PregEndIndi PregEndMonth PregEndSD) ;
      by PatientID_OPA;
      if liveb ;
      if numlivebirths=1 then DeliveryCount = 1;
      else if numlivebirths=2 then 
         do;
         DeliveryDiff = max(DOSNew - FirstLiveBirthDate,-1);
         if DeliveryDiff >=180 then DeliveryCount = 2;  *...assigns all claims after 180 days to 2nd live birth event for those with 2...[PH];
         else DeliveryCount = 1;
         end;
      %*...This section converts the numeric deliverycount variable to a character variable;
      %*...and concatenates Patient ID and DeliveryCountChar to create the new Patient ID;
      %*...for women with a live birth in the measurement year...[PH];
      DeliveryCountChar = put(DeliveryCount, 1.);                
      PatientID_orig = PatientID_OPA;
      PatientID_OPA = trim(right(substr(trim(right(PatientID_OPA)),1,length(trim(PatientID_OPA))-1)||DeliveryCountChar));
      run;

   %*...Sort ContraceptiveClaimsNewID by Patient ID and date of service to prepare to create a one record per person dataset;
   proc sort data = ContraceptiveClaimsNewID; 
      by PatientID_OPA DOSNew;
      run;

   %*...Create a one record per person dataset called ContraceptiveClaimsNewIDOne;
   data ContraceptiveClaimsNewIDOne; 
      set ContraceptiveClaimsNewID;
      by PatientID_OPA DOSNew;
      if first.PatientID_OPA;
      run;

   *...This macro creates one record per person datasets with indicators for these conditions: infecund, pregnant, non-live-birth, live birth, end of pregnancy;
   %macro exclude_pp(condition,firstlast); 
      *...Macro parameters: 1) condition and 2) which claim to select (first or last)...[PH];
      options mprint=1;
      *...Create datasets with only those patients having the condition;
      data &condition.;
         set ContraceptiveClaimsNewID;
         where &condition.=1;
         run;

      *...Sort claims dataset by Patient ID and date of service to prepare to create a one record per person dataset with women who have condition;
      proc sort data = &condition.; 
         by PatientID_OPA DOSNew;
         run;

      *...Create a one record per person dataset with a condition indicator;
      data &condition.One (keep = PatientID_OPA &condition.Indi &condition.Month &condition.SD);
         set &condition.;
         format &condition.SD yymmddd10.;
         by PatientID_OPA DOSNew;
         if &firstlast..PatientID_OPA;
         &condition.Indi = 1;
         &condition.Month = month;
         &condition.SD = DOSNew;
         run;

      *...Sort condition indicator dataset to prepare to merge;
      proc sort data = &condition.One; 
         by PatientID_OPA;
         run;
   %mend;
   %exclude_pp(Pregnant,first);
   %exclude_pp(NonLiveBirth,last);
   %exclude_pp(PregEnd,last);

   %*...Create a dataset with pregnancy and end of pregnancy indicators;
   data PregnancyIndicator_pp 
        (keep = PatientID_OPA PregnantIndi PregnantMonth PregnantSD 
                NonLiveBirthIndi NonLiveBirthMonth NonLiveBirthSD 
                PregEndIndi PregEndMonth PregEndSD); 
      merge PregnantOne NonLiveBirthOne PregEndOne
            ContraceptiveClaimsNewIDOne (in = a keep=patientID_OPA) ;
      by PatientID_OPA;
      if a;
      run;
   proc sort data = ContraceptiveClaimsNewID ; 
      by PatientID_OPA;
      run;

   %*...Sort pregnancy indicator dataset to prepare to merge with multiple records per person full dataset;
   proc sort data = PregnancyIndicator_pp; 
      by PatientID_OPA;
      run;

   %*...Merge pregnancy indicator dataset and multiple records per person full dataset to give women pregnancy indicators;
   %*...Create exclude variable (exclude = 1 for infecund, exclude = 2 for still pregnant, exclude = 3 for pregnancies ending in last 3 months of the year);
   data ContraceptiveClaimsNewID;
      merge PregnancyIndicator_pp 
            ContraceptiveClaimsNewID ; 
      by PatientID_OPA;
      run;

   %*...Delete intermediate datasets;
   proc datasets lib = work nolist;
      delete LiveBirth_pp livebirthdates livebirthfirst livebirthlast PregnantOne 
             NonLiveBirthOne PregEndOne Pregnant NonLiveBirth PregEnd pregnancyindicator_pp ;
      quit;

   %*...Limit ContraceptiveClaimsNewID to just live births;
   data LiveBirthB; 
      set ContraceptiveClaimsNewID;
      where LiveBirth = 1;
      run;

   %*...Sort live birth claims dataset by Patient ID and date of service variable to prepare to create a dataset with women who had a live birth in the measurement year;
   proc sort data = LiveBirthB; 
      by PatientID_OPA DOSNew;
      run;

   %*...This sub-macro creates LiveBirthSD# variables for each number(#) entered by user in dayspp_list;
   %macro genppvars;
   %let j = 1;
   %do %while (%scan(&dayspp_list, &j) ne );
      %let n = %scan(&dayspp_list, &j);
      format LiveBirthSD&n. yymmddd10.;                                                                
      LiveBirthSD&n. = intnx('day',LiveBirthSD,&n.);  %*...LiveBirthSD# is date of # days postpartum;                                                     
      %let j = %eval(&j + 1);
   %end;
   %mend genppvars;

*...FD: original codes in v1.03a, uses last live birth claim to flag live birth date of service. No longer in use;
/*
   %*...Create a one record per person dataset to determine dates of live birth;
   data LiveBirthOne (keep = PatientID_OPA LiveBirthIndi LiveBirthMonth LiveBirthSD: LiveBirthAge LiveBirthAgeGroup); 
      set LiveBirthB; 
      format LiveBirthSD yymmddd10. LiveBirthAgeGroup agegroupf.;
      attrib LiveBirthAgeGroup label='Age Group at Date of Live Birth';
      by PatientID_OPA DOSNew;
      if last.PatientID_OPA;
      LiveBirthIndi = 1;
      LiveBirthMonth = month;
      LiveBirthSD = DOSNew;

      %genppvars;

      LiveBirthAge = AGE_CALC;
      if 15 <= AGE_CALC < 21 then LiveBirthAgeGroup = 1;
      else if 21 <= AGE_CALC < 45 then LiveBirthAgeGroup = 2;
      run;      
*/

%*...FD EDIT === UPDATE Feb 2019 === ;
%*...DF: This version uses the first date of live birth claim to flag live birth date of service in order to capture all postpartum contraceptive methods;
%*...where as keep using the last date of live birth claim to flag live birth month in order to flag women who gave birth twice in the year and had the 2nd birth in Oct-Dec;
%*...Reason for change: the updated lookup table uses both diagnosis and procedure codes to flag live birth, whereas the original lookup table uses procedure codes only;
%*...Since diagnosis code Z37.0 (single live birth) is used for any claim that is related to the live birth including some postpartum care;
%*...multiple dates can be flagged as live birth using the updated lookup table;
%*...The previous program uses the last date of live birth-flagged claim as live birth service date, which may incorrectly flag live birth service date if a woman had;
%*...multiple dates flagged as live birth;
%*...For women who had contraceptive care on the same day of birth (common for sterilization), postpartum contraceptive provision is missed-counted because contraceptive provision date;
%*...is flagged before the incorrect live birth service date;
%*...The updated program changed the live birth service date to the first date to solve this issue;


   %*...Create a one record per person dataset to determine month of live birth using the last record service date;
   data LiveBirthB_2; 
      set LiveBirthB; 
      by PatientID_OPA DOSNew;
      if last.PatientID_OPA;
      LiveBirthIndi = 1;
      LiveBirthMonth = month;
      run;

   %*...Create another one record per person dataset to determine date of live birth using the first record service day;
   data LiveBirthB_3 (keep = PatientID_OPA AGE_CALC LiveBirthSD); 
      set LiveBirthB; 
      format LiveBirthSD yymmddd10. ;
      by PatientID_OPA DOSNew;
      if first.PatientID_OPA;
      LiveBirthSD = DOSNew;
      run;

   %*...Merge 2 datesets together to get both month and date of live birth;
   data LiveBirthOne (keep = PatientID_OPA LiveBirthIndi LiveBirthMonth LiveBirthSD: LiveBirthAge LiveBirthAgeGroup);
      merge LiveBirthB_2 LiveBirthB_3;
      by PatientID_OPA;
      format LiveBirthAgeGroup agegroupf.;  
      attrib LiveBirthAgeGroup label='Age Group at Date of Live Birth';

      %genppvars;

      LiveBirthAge = AGE_CALC;
      if 15 <= AGE_CALC < 21 then LiveBirthAgeGroup = 1;
      else if 21 <= AGE_CALC < 45 then LiveBirthAgeGroup = 2;
      run;      

   %*...Sort ContraceptiveClaimsNewIDOne by Patient ID to prepare to merge with LiveBirthOne; 
   proc sort data = ContraceptiveClaimsNewIDOne; 
      by PatientID_OPA;
      run;

   %*...Sort LiveBirthOne by Patient ID to prepare to merge with ContraceptiveClaimsNewIDOne;
   proc sort data = LiveBirthOne; 
      by PatientID_OPA;
      run;

   %*...Create dataset live birth indicators;
   data LiveBirthIndicator (keep = PatientID_OPA LiveBirthSD LiveBirthIndi LiveBirthMonth LiveBirthSD LiveBirthSD: LiveBirthAge 
      LiveBirthAgeGroup); 
      merge ContraceptiveClaimsNewIDOne (in = a) LiveBirthOne;
      by PatientID_OPA;
      if a = 1;
      run;

   %*...Sort live birth indicator dataset to prepare to merge with multiple records per person full dataset;
   proc sort data = LiveBirthIndicator; 
      by PatientID_OPA;
      run;

   %*...Sort multiple records per person full dataset to prepare to merge with pregnancy indicator dataset;
   proc sort data = ContraceptiveClaimsNewID;
      by PatientID_OPA;
      run;

   %*...Merge live birth indicator dataset and multiple records per person full dataset to give women pregnancy indicators.
         Create a postpartum exclusion variable defining deliveries with a known miscarriage, ectopic pregnancy, stillbirth, or induced abortion 
         (Exclude_pp = 1 Table PCU-B: Codes indicating a known miscarriage, ectopic pregnancy, stillbirth, or induced 
         abortion) and women who had a live birth in the last 3 months of the year and may not have had an opportunity to receive a 
         contraceptive method (Exclude_pp = 2);
   data ContraceptiveClaimsNewIDPreg;
      %*...reverse merge order so LiveBirthIndicators are merged to every claim...[PH];
      merge LiveBirthIndicator ContraceptiveClaimsNewID (drop=LiveBirthSD LiveBirthIndi LiveBirthMonth LiveBirthSD LiveBirthSD: ); 
      by PatientID_OPA;
      if NonLiveBirth = 1 then Exclude_pp = 1; 
      else if LiveBirthIndi = 1 and LiveBirthMonth > 9 then Exclude_pp = 2;
      else Exclude_pp = 0;
      run;

   %*...Delete intermediate datasets;
   proc datasets lib = work nolist;
      delete ContraceptiveClaimsNewID ContraceptiveClaimsNewIDOne LiveBirthOne LiveBirthIndicator LiveBirthB;
      quit;

   %*...Create dataset ExcludeIndi with only records that meet the two exclusion criteria;
   data ExcludeIndi; 
      set ContraceptiveClaimsNewIDPreg;
      where Exclude_pp > 0;
      run;

   %*...Sort dataset ExcludeIndi to prepare to create a one record per person dataset that will create an exclusion indicator variable;
   proc sort data = ExcludeIndi; 
      by PatientID_OPA Exclude_pp;
      run;

   %*...Create a one record per person indicator dataset with an exclusion indicator variable;
   data ExcludeIndiOne (keep = PatientID_OPA ExcludeIndi Exclude_pp); 
      set ExcludeIndi; 
      by PatientID_OPA Exclude_pp;
      if last.PatientID_OPA;
      ExcludeIndi = 1;
      run;

   %*...Sort ExcludeIndiOne dataset by Patient ID to prepare to merge with multiple record per person dataset ContraceptivesAge;
   proc sort data = ExcludeIndiOne;
      by PatientID_OPA;
      run;

   %*...Sort ContraceptivesAge dataset by Patient ID to prepare to merge with ExcludeIndiOne dataset;
   proc sort data = ContraceptiveClaimsNewIDPreg;
      by PatientID_OPA;
      run;

   %*...Merge ExcludeIndiOne with ContraceptivesAges to create an exclusion indicator;
   data ContraceptivesAgeExclude; 
      merge ContraceptiveClaimsNewIDPreg (in = a drop=exclude_pp) ExcludeIndiOne ; *...new change: drop exclude_pp for correct merge to many...[PH];
      by PatientID_OPA;
      if a = 1;
      if exclude_pp=. then exclude_pp=0;
      run;

   %*...Delete intermediate datasets;
   proc datasets lib = work nolist;
      delete ContraceptiveClaimsNewIDPreg ExcludeIndiOne ExcludeIndi;
      quit;

   %*...This macro creates a flag variable (flag_pp#) that flags claims occuring within # days postpartum;
   %macro genppflags;
      %let i = 1;
      %do %while (%scan(&dayspp_list, &i) ne );
         %let n = %scan(&dayspp_list, &i);
         if (LiveBirthIndi = 1 and (LiveBirthSD <= DOSNew <= LiveBirthSD&n.) and ExcludeIndi ^=1) then flag_pp&n. = 1;
         if (LiveBirthIndi = 1 and (LiveBirthSD <= DOSNew <= LiveBirthSD&n.) ) then flagall_pp&n. = 1; %***PH Edit: add flagall_pp&n. for exclusion freqs...[PH];
         %let i = %eval(&i + 1);
      %end;
   %mend genppflags;

   %*...Flag claims that occurred within n days of a live birth to look at postpartum contraception in the n days post-delivery (n specified by user in dayspp_list);
   data NumeratorData_pp (where=(15 <= AGE_CALC < 45));
      set ContraceptivesAgeExclude;
      %genppflags;
      run;

   %*...Create person-level postpartum flag indicators for the denominator data;
   %macro ppflagsindi;
      %let i = 1;
      %do %while (%scan(&dayspp_list, &i) ne );
      %let n = %scan(&dayspp_list, &i);

      data ppflag&n.Indi;
         set NumeratorData_pp;
         where flag_pp&n. = 1;
         run;

      proc sort data = ppflag&n.indi;
         by PatientID_OPA flag_pp&n.;
         run;

      data ppflag&n.IndiOne (keep = PatientID_OPA flag_ppindi_&n.);
         set ppflag&n.indi;
         by PatientID_OPA flag_pp&n.;
         if last.PatientID_OPA;
         flag_ppindi_&n. = 1;
         run;

      proc sort data = ppflag&n.IndiOne;
         by PatientID_OPA;
         run;

      %*This section makes dataset to keep all within timeframe -- even the exclusions (instead of flag_ppindi_* above) ...[PH];
      data ppflagall&n.Indi;
         set NumeratorData_pp;
         where flagall_pp&n. = 1;
         run;

      proc sort data = ppflagall&n.indi;
         by PatientID_OPA flagall_pp&n.;
         run;

      data ppflagall&n.IndiOne (keep = PatientID_OPA flagall_ppindi_&n.);
         set ppflagall&n.indi;
         by PatientID_OPA flagall_pp&n.;
         if last.PatientID_OPA;
         flagall_ppindi_&n. = 1;
         run;

      proc sort data = ppflagall&n.IndiOne;
         by PatientID_OPA;
         run;
      %*END section to keep all within timeframe -- even the exclusions (instead of flag_ppindi_* above) ...[PH];

      proc sort data = NumeratorData_pp;
         by PatientID_OPA;
         run;

      data NumeratorData_pp;
         merge NumeratorData_pp ppflag&n.IndiOne ppflagall&n.IndiOne;
         by PatientID_OPA;  %*...Keep this flagall_ppindi_&n. across excluded, for reporting...[PH];
         attrib flag_ppindi_&n. label = "Person-level flag for inclusion in &n. days postpartum measure calculation"
                flagall_ppindi_&n. label = "Person-level flag for &n. days postpartum claims"  
                flag_pp&n. label = "Claims-level flag for inclusion in &n. days postpartum numerator calculations";
         run;

      %*...Sort the postpartum dataset by Patient ID and date of service to prepare to create a one record per person dataset that will be used for the denominator;
      proc sort data = NumeratorData_pp; 
         by PatientID_OPA DOSNew;
         run;

      %*...Create a one record per person dataset to determine how many women were excluded;
      data numeratordataone_pp; 
         set NumeratorData_pp;
         by PatientID_OPA DOSNew;
         if last.PatientID_OPA;
         run;

      %*...Delete intermediate datasets;
      proc datasets lib = work nolist;
         delete ppflag&n.Indi ppflag&n.IndiOne ppflagall&n.Indi ppflagall&n.IndiOne ;
         run; quit;

      %let i = %eval(&i + 1);
      %end;
   %mend ppflagsindi;
   %ppflagsindi;


   %*...Create a one record per person dataset to use in final merge;
   data DenominatorDataOne_pp;
      set NumeratorData_pp; 
      by PatientID_OPA DOSNew;
      if first.PatientID_OPA;
      attrib exclude_all label = 'Exclusions: All Women (0=include | 1=infecund | 2=still pregnant | 3=preg ended Oct-Dec)'
             exclude_pp label = 'Exclusions: Postpartum Women (0=include | 1=miscarriage, ectopic, stillbirth, induced abortion | 2=preg ended Oct-Dec)';
      run;

      %*...Delete intermediate datasets;
      proc datasets lib = work nolist;
         delete ContraceptivesAgeExclude ;
      quit;

   %end; %*...end entire postpartum section (for postpartum=y)...;

%mend postpartum;
%postpartum;

%*...Delete intermediate datasets;
proc datasets lib = work nolist;
   delete ContraceptivesAge Contraceptivespreg ContraceptiveClaims ContraceptiveClaimsOne Livebirthb_2 Livebirthb_3;
quit;
run;












